from rest_framework import serializers
from .models import Package, ItineraryDay


class ItineraryDaySerializer(serializers.ModelSerializer):
    class Meta:
        model = ItineraryDay
        fields = ["id", "day", "title", "description", "price_per_adult", "price_per_child"]


class PackageSerializer(serializers.ModelSerializer):
    itinerary_days = ItineraryDaySerializer(many=True)
    total = serializers.DecimalField(max_digits=12, decimal_places=2, read_only=True)

    class Meta:
        model = Package
        fields = [
            "id",
            "package",
            "adults",
            "children",
            "nights",
            "days",
            "total",
            "created_at",
            "itinerary_days",
        ]
        read_only_fields = ["id", "total", "created_at"]

    def create(self, validated_data):
        itinerary_data = validated_data.pop("itinerary_days", [])
        package = Package.objects.create(**validated_data)

        # Create itinerary days
        for day_data in itinerary_data:
            ItineraryDay.objects.create(package=package, **day_data)

        # Calculate total after saving
        package.total = package.calculate_total()
        package.save()
        return package

    def update(self, instance, validated_data):
        itinerary_data = validated_data.pop("itinerary_days", None)

        # Update package fields
        for attr, value in validated_data.items():
            setattr(instance, attr, value)
        instance.save()

        # Update or replace itinerary days
        if itinerary_data is not None:
            instance.itinerary_days.all().delete()  # clear existing
            for day_data in itinerary_data:
                ItineraryDay.objects.create(package=instance, **day_data)

        # Recalculate total
        instance.total = instance.calculate_total()
        instance.save()
        return instance
