from django.db import models
from django.contrib.auth import get_user_model

User = get_user_model()


class Experience(models.Model):
    CATEGORY_CHOICES = [
        ("tour", "Tour"),
        ("adventure", "Adventure"),
        ("cultural", "Cultural"),
        ("wildlife", "Wildlife"),
        ("other", "Other"),
    ]

    title = models.CharField(max_length=255)
    description = models.TextField()
    category = models.CharField(
        max_length=50, choices=CATEGORY_CHOICES, default="other"
    )
    image = models.ImageField(
        upload_to="experiences/%Y/%m/%d/",  # Date-based folders
        blank=True,
        null=True
    )
    price = models.DecimalField(max_digits=10, decimal_places=2)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.title} ({self.category})"


class Booking(models.Model):
    STATUS_CHOICES = [
        ("pending", "Pending"),
        ("confirmed", "Confirmed"),
        ("cancelled", "Cancelled"),
    ]

    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="bookings")
    experience = models.ForeignKey(
        Experience, on_delete=models.CASCADE, related_name="bookings"
    )
    booking_date = models.DateTimeField(auto_now_add=True)
    number_of_people = models.PositiveIntegerField(default=1)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default="pending")
    total_price = models.DecimalField(max_digits=10, decimal_places=2, default=0.00)

    def save(self, *args, **kwargs):
        if self.experience and self.number_of_people:
            self.total_price = self.experience.price * self.number_of_people
        super().save(*args, **kwargs)

    def __str__(self):
        return f"Booking #{self.id} - {self.user.email} - {self.experience.title}"
